#############################################################################
#
#              GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/GnutonVM.py,v $
# Version:     $RCSfile: GnutonVM.py,v $ $Revision: 1.3 $
# Copyright:   (C) 1997-1999, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""

the virtual machine class

"""

#############################################################################

import string

from   GnutOS.ByteCodes     import inst_table
from   GnutOS.Stack         import Stack
from   GnutOS.Classes       import *


#############################################################################

class VM_ImmediateTooLarge:
    pass

class VM_BadType:
    pass

class VM_LiteralIndexOutOfRange:
    pass

class VM_NoStackFrameForReturn:
    pass

#############################################################################

class StackFrame:
    """NewtonScript stack frame class."""

    def __init__(self):

	self._func = None         # current function object
	self._pc = 0              # byte index (from 0) to instructions
	self._sp = 0              # index of last pushed stack value
	self._locals = None       # local variables frame
	self._rcvr = None         # receiver (for msg send)
	self._impl = None         # implementor (for msg send)

	return


#############################################################################

class Iterator:
    """NewtonScript iterator."""

    def __init__(self, target, deeply):

	self._target = target
	self._frame = hasattr(self._target, "_map")

	self._deeply = deeply
	if not self._frame and self._deeply:
	    raise VM_DeepIteratorForNonFrame()

	if not self._frame:
	    self._list = self._target.values()

	else:
	    self._list = self._target._slots
	    if deeply:
		t = self._target
		while hasattr(t, "_map") and t._map.has_key("_proto"):
		    t = t._slots[t._map["_proto"]]
		    self._list.extend(t._slots)

	self._current = 0
	self._done = 0
	return

    def next(self):
	if self._done:
	    raise VM_IteratorNextWhenDone()
	    
	self._current = self._current + 1

	if self._current > len(self._list):
	    self.done = 1

	return self.done


    def aref(self, arg):
	if arg == 0:
	    return tag   #fixme!

	elif arg == 1:
	    return self._list[self._current]

	elif arg == 3 and self._deeply:
	    return len(self._list)

	elif arg == 5:
	    return len(self._target)

	else:
	    raise VM_BadArefArgument(arg)


    def done(self):
	return self._done


#############################################################################

class GnutonVM:
    """Virtual Machine Class

    This class creates an instance of an interpreter; multiple
    instances can share a memory array, allowing multiple threads to
    be executed within a Python program.

    The interpreter state consists a a reference to the memory array,
    an instruction pointer, and a stack pointer.

    The memory is represented as a Python string: an array of octets.
    A range of functions translate values from their in-memory
    representation into their Python representation and back again,
    enabling the implementation of the bytecodes in Python.

    The stack is represented as a stack of Python objects, each
    representing a single language object.  This prevents us from
    having to marshal a on and off the stack.

    """

    def __init__(self, ref_am):
	"""Create a new interpreter."""

	self._am = ref_am
	self._debug = 1

	#-- stacks
	self._stack = Stack()
	self._callstack = Stack()
	self._excstack = Stack()

	self._fr = None           # current stack frame
	self._intr = 0            # single step interrupt flag

	#-- initialise instruction table
	self._lst_inst = map(self._opcode_method, inst_table)

	return


    def debug(self, *args):
	if self._debug:
	    print "--", string.join(map(str, args))


    def load(self, func_obj):
	"""Start execution.

	*func_obj*  -- (frame) function object to execute
	Returns     -- (none)

	The *func_obj* should be a frame with class: 'CodeBlock, etc,
	as described in section 2 of the Newton Formats doc.

	Execution will continue until a "return" from the specified
	function, a VM exception, end of memory or the interrupt flag
	is set."""

	#-- initialise execution context (frame)
	self._fr = StackFrame()
	self._fr._func = func_obj                     # the function frame
	self._fr._pc = 0                              # reset program counter
	self._fr._stack = Stack()                     # initialise the stack
	self._fr._rcvr = None
	self._fr._impl = None

	#-- set locals frame, checking for NIL
	if self._fr._func["argFrame"].PrimClassOf() == "immediate ref" and \
	   self._fr._func["argFrame"].isNil():
	    self._fr._locals = self._fr._func["argFrame"]
	else:
	    self._fr._locals = self._fr._func["argFrame"].target()

	#-- get instructions
	self._bcs = self._fr._func["instructions"].target().getData()
	self._plen = len(self._bcs)

	#for c in self._bcs:
	#    print hex(ord(c) & 0xf8),
	#print

	return


    def interrupt(self):
	"""Interrupt the interpreter."""
	self._intr = 1
	return


    def run(self):
	while self._fr._pc < self._plen and not self._intr:
	    self.step()
	return


    def step(self):
	"""Execute a single instruction.

	Return -- (none)

	Execute a single instruction and return the "ip" of the next
	instruction."""

	inst = self._fetch()                 # fetch bytecodes
	op, arg, pci = self._decode(inst)    # decode instruction
	self._fr._pc = self._fr._pc + pci    # increment PC

	apply(op, (self, arg))               # execute bytecode
	return


    def _fetch(self):
	"""Fetch the next instruction.

	Returns a string containing the next 3 (normally) bytes from
	the instructions binary.  If there are fewer than three bytes
	remaining of the instructions, only those remaining are
	returned."""

	inst = self._bcs[self._fr._pc: self._fr._pc + 3]
	#self._fr._pc = self._fr._pc + 3

	return inst


    def _decode(self, inst):
	"""Decode an instruction.

	"inst"          3 character string from memory array
	Return value    (op, arg) method reference and argument
	
	Decode the supplied instruction, and set the instruction
	pointer back 2 if it is a single byte opcode.  Return a
	reference to the method implementing the instruction, and the
	converted argument."""

	#-- separate each byte code
	i1 = inst[:1] and ord(inst[:1]) or 0
	i2 = inst[1:2] and ord(inst[1:2]) or 0
	i3 = inst[2:3] and ord(inst[2:3]) or 0

	ok = 0      # error flag
	pci = 0     # program counter increment

	#-- lookup instruction table
	for (bc, oc, ds, msk, b1, b2, b3, n, m) in self._lst_inst:
	    if i1 & msk == b1:
		if bc == 3 and i2 == b2 and i3 == b3:
		    meth = m
		    pci = 3
		    arg = 0
		    print "%04x: %02x %s" % (self._fr._pc, b1, n),

		elif i1 & ~msk < 7:
		    meth = m
		    pci = 1
		    arg = i1 & ~msk
		    print "%04x: %02x %23s (0x%04x == %s)" % (self._fr._pc, b1, n, arg, hword2text(arg))

		elif i1 & ~msk == 7:
		    meth = m
		    pci = 3
		    arg = (i2 << 8) + i3
		    print "%04x: %02x %23s (0x%04x == %s)" % (self._fr._pc, b1, n, arg, hword2text(arg))

		else:
		    continue

		ok = 1
		break

	if not ok:
	    raise VMError, "Illegal instruction %s %s %s" % (hex(i1), hex(i2), hex(i3))

	return (m, arg, pci)


    def _opcode_method(self, t_op):
	"""Return opcode tuple with method appended."""

	bc, oc, ds, m, b1, b2, b3, n = t_op
	ref_meth = self.__class__.__dict__["bc_%s" % n]
	return (bc, oc, ds, m, b1, b2, b3, n, ref_meth)


    def _proto_lookup(start, name):
	"""Follow only the proto inheritence chain to find value"""

	current = start

	while current:
	    if HasSlot(current, name):
		return GetSlot(current, name)
	    current = GetSlot(current, "_proto")

	raise KeyError


    def _lexical_lookup(start, name):
	"""Follow only the nested argFrame chain to find value"""

	current = start

	while current:
	    if HasSlot(current, name):
		return GetSlot(current, name)
	    current = GetSlot(current, "_nextArgFrame")

	raise KeyError


    def _full_lookup(start, name):
	"""Follow both _parent and _proto chains to find value"""

	left = start
	while left:
	    current = left
	    while current:
		if HasSlot(current, name):
		    return GetSlot(current, name)
		current = GetSlot(current, "_proto")

	    left = GetSlot(left, "_parent")

	raise KeyError


    def _assignment(start, name, value):
	left = start
	while left:
	    current = left
	    while current:
		if HasSlot(current, name):
		    SetSlot(left, name, value)
		    return

		current = GetSlot(current, "_proto")
	    left = GetSlot(left, "_parent")

	raise KeyError


    def _lexical_assignment(start, name, value):
	current = start
	while current:
	    if HasSlot(current, name):
		SetSlot(current, name, value)
		return

	    current = GetSlot(current, "_nextArgFrame")

	raise KeyError

    

#############################################################################
#
#   simple functions (ie. "A" field is 0, "B" field ranges from 0 to 7
#

    def bc_pop(self, arg):          # pop (B = 0) (0x0)
	"""Removes the top item from the stack.

	"arg"           (not used)."""

	self._stack.pop()
	return


    def bc_dup(self, arg):          # dup (B = 1) (0x1)
	"""Duplicates the top item of the stack.

	"arg"           (not used)."""

	self._stack.push(self._stack.top())
	return


    def bc_return(self, arg):       # return (B = 2) (0x2)
	"""Exits from a function.

	*arg*    -- (not used)
	Returns  -- (none)

	By convention, the top item on the stack is the return value
	of the routine.  It is possible to return from a routine with
	more items on the stack (or none at all), however the current
	Apple NewtonScript compiler ensures that the functions that it
	produces always leave one and only one value on the stack."""

	if self._callstack.size() < 1:
	    return VM_NoStackFrameForReturn()

	self._fr = self._callstack.pop()
	return


    def bc_push_self(self, arg):         # push-self (B = 3) (0x3) 
	"""Push reference to frame containing current function.

	*arg*    -- (not used)
	Returns  -- (none)

	Pushes a reference to the frame containing the current
	function onto the stack.  In other words this pushes the value
	represented by the NewtonScript value self."""

	self._stack.push(self._fr._rcvr)
	return


    def bc_set_lex_scope(self, arg):      # set-lex-scope (B = 4) (0x4)
	"""Create closure.

	*arg*    -- (not used)
	Returns  -- (none)

	Pops a function object from the stack, constructs a closure
	from the function object and pushes the closure onto the
	stack.

	A closure is a function object with saved values for the outer
	lexical environment, the current receiver, and the current
	implementer.  These values are saved in the locals frame of
	the function object in the _nextArgFrame, _parent and
	_implementor slots. The original function object and its
	locals frame are cloned, and the values are filled in."""

	#fixme: check func obj on stack is 'CodeBlock
	fn = Clone(self._stack.pop())
	af = Clone(fn["argFrame"])

	af["_nextArgFrame"] = self._fr._locals
	af["_parent"] = self._fr._rcvr
	af["_implementor"] = self._fr._impl
	fn["argFrame"] = af

	self._stack.push(fn)
	return


    def bc_iter_next(self, arg):      # iter-next (B = 5)
	"""Do next iteration in foreach.

	*arg*    -- (not used)
	Returns  -- (none)

	Pops an iterator from the stack and advances it to the next
	slot."""

	iter = self._stack.pop()
	iter.next()
	return


    def bc_iter_done(self, arg):  # iter-done (B = 6)
	"""Checks if iteration complete.

	*arg*    -- (not used)
	Returns  -- (none)

	Pops an iterator from the stack.  If it is exhausted, pushes
	TRUE onto the stack; otherwise, pushes NIL onto the stack."""

	iter = self._stack.pop()
	if iter.done():
	    self._stack.push(TRUE)
	else:
	    self._stack.push(NIL)
	return


    def bc_pop_handlers(self, arg):     # pop-handlers (B = 7)
	"""End exception.

	*arg*    -- (not used)
	Returns  -- (none)

	Removes the most recent exception handler context."""

	self._excstack.pop()
	return


#############################################################################
#
#   parameterised functions (ie. "B" field is 16 bits
#

    def bc_push(self, arg):          # push (A = 3)
	"""Push "arg"th element of literals array on stack.

	*arg*           (integer) index into literals array
	Return value    (none)

	The argument is a zero-based index into the literals array."""

	lits = self._fr._func["literals"].target()

	if arg >= len(lits):
	    raise VM_LiteralIndexOutOfRange(arg, len(lits))

	self._stack.push(lits[arg])
	self.debug(lits[arg])
	return


    def bc_push_constant(self, arg):          # push-constant (A = 4)
	"""Push immediate value on stack.

	*arg*    -- (Ref) integer, char, magic or special.  see below.
	Returns  -- (none)

	**Exceptions**
	VM_ImmediateTooLarge  -- value of the immediate ref exceeded 16 bits 
	VM_BadType            -- cannot push a pointer ref as constant

	This instruction is used to push any immediate value that can
	be represented in 16 (sign extended) bits or less.  Values
	that require the full 32 bits are stored as literals and
	pushed using BC_PUSHLITERAL.

	Some examples of the use of this instruction:

	20		push 0
	22		push Nil
	23		push @0
	24		push 1
	27 00 1A	push True
	27 02 86	push $(
	27 FF F8	push -2

	Note that the value of B is *always* interpreted as signed."""

	#-- mask type and extract value
	ref_type = arg & 3
	ref_value = arg >> 2

	#-- create ref object
	if ref_type == 0:
	    ref = Integer(ref_value)
	elif ref_type == 1:
	    raise VM_BadType("cannot push pointer as constant")
	elif ref_type == 2:
	    ref = Special(ref_value)
	elif ref_type == 3:
	    ref = Magic(0, ref_value & 0xfff)

	#-- push
	self._stack.push(ref)
	return


    def bc_get_var(self, arg):   # A = 15
	"""Push "arg"th element of arguments frame on stack.

	This bytecode is commonly used to push either the value of an
	argument to the function or that of a local variable.

	Example:

	func( foo ) return foo translates to:

	73	push value of foo (remember argument slots start
				   after the three fixed slots in
				   argFrame, hence the index is 3 not
				   0).
	02	Return

	"""

	if self._fr._locals.isNil() or arg >= len(self._fr._locals):
	    raise VM_LocalIndexOutOfRange(arg, self._fr._locals)

	self._stack.push(self._fr._locals.nth(arg))
	self.debug(self._fr._locals.nth(arg))
	return


    def bc_get_path(self, arg):
	"""

	This takes a frame and the name of a slot in that frame and
	pushes the value contained in the slot.  If the slot name is a
	single name then it is represented by a symbol reference.  If
	the slot name is a path expression (e.g. foo.bar) then it is
	represented by an array of class pathExpr with one element per
	section of the path.  Each element is then a reference to the
	symbol for that part of the path.

	Example:

	return self.foo.bar translates to:

	03	push self
	18	push '[pathExpr: 'foo, 'bar] (assuming it to be
					      the first literal)
	91	Get slot value
	02	Return

	"""

	pass


    def bc_find_var(self, arg):
	"""

	This bytecode pushes onto the stack the value contained in a
	variable or slot from outside of the function.  The name of
	the variable to push is given by literal "arg" in the
	function's literals array; this literal should be a symbol
	reference.

	Example:

	length( functions ) translates to:

	70	push value of functions (assuming 'functions is the
					 first entry in	the literals
					 array of the containing CodeBlock)
	C7 00 12	Return the length of functions.

	"""

	sym = 0
	pass


    def bc_set_path(self, arg):
	"""

	This assigns the value Y to the slot SlotName of frame X,
	i.e. X.SlotName := Y.  As with 221 (push slot value) the slot
	name is represented either by a simple symbol reference or a
	pathExpr array for a complex path expression.

	"""

	pass


    def bc_set_var(self, arg):
	"""

	Value X is assigned to the "arg"th slot of the CodeBlock's
	argFrame frame.  In other words, this bytecode is usually used
	to set the value of a local variable.

	Example:

	local x := '[] translates to:

	18	push '[] (assuming it to be the first literal)
	A7 00 10	Assign to x (assuming it to be 16th slot in argFrame)

	"""

	pass


    def bc_find_and_set_var(self, arg):
	"""

	Value X is assigned to a variable or slot from outside of the
	function.  The name of the variable to assign to is given by
	literal N in the function's literals array; this literal
	should be a symbol reference.

	"""

	pass


    def bc_add(self, arg):
	"""Addition of integers and floats.

	x = pop(); y = pop(); push (x+y)

	The numbers can be integers or floats.

	The 30N bytecode series encodes a number of different
	low-level system function calls.  Any NewtonScript operator or
	built-in function not explicitly mentioned here is implemented
	by using a standard function call (bytecode 05N).  This first
	bytecode of the series implements addition (of integers or
	floats).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x+y)

	return


    def bc_subtract(self, arg):
	"""Subtraction of integers or floats.

	x = pop(); y = pop(); push (x-y)
	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x-y)

	return


    def bc_multiply(self, arg):
	"""Multiply two numbers.

	x = pop(); y = pop(); push (x*y)
	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(x*y)

	return


    def bc_divide(self, arg):
	"""Divide two numbers with float result.

	x = pop(); y = pop(); push (x/y)

	This implements division (of integers or floats) with a float
	result, i.e.  it corresponds to the <#007F>/ operator in
	NewtonScript.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(float(x)/y)

	return


    def bc_div(self, arg):
	"""

	This implements division of integers with an integer result,
	i.e. it corresponds to the div operator in
	NewtonScript.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- check types

	#-- push result
	self._stack.push(int(float(x)/y))

	return


    def bc_not(self, arg):
	"""

	This takes any value X and returns not X.  Not Nil is True,
	not anything else is Nil.

	"""

	#-- pop arguments
	x = self._stack.pop()

	#-- test
	if x == NS_NIL:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_equals(self, arg):
	"""


	This compares X and Y for equality and pushes a Boolean result (True
	or Nil).

	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x == y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)

	return


    def bc_not_equals(self, arg):
	"""

	This compares X and Y for inequality and pushes a Boolean
	result (True or Nil).

	"""

	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x != y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_less_than(self, arg):
	"""

	This checks if X is less than Y and pushes a Boolean result
	(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x < y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_greater_than(self, arg):
	"""

	This checks if X is greater than Y and pushes a Boolean result
	(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x > y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_less_or_equal(self, arg):
	"""

	This checks if X is less than or equal to Y and pushes a
	Boolean result(True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x <= y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_greater_or_equal(self, arg):
	"""

	This checks if X is greater than or equal to Y and pushes a
	Boolean result (True or Nil).

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	#-- test
	if x >= y:
	    self._stack.push(NS_TRUE)
	else:
	    self._stack.push(NS_NIL)


    def bc_bit_and(self, arg):
	"""

	This performs a binary and of the two integers X and Y.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	self._stack.push(x & y)


    def bc_bit_or(self, arg):
	"""

	This performs a binary or of the two integers X and Y.

	"""
	#-- pop arguments
	x = self._stack.pop()
	y = self._stack.pop()

	self._stack.push(x | y)


    def bc_bit_not(self, arg):
	"""

	This performs a binary not of the integer X.

	"""
	#-- pop arguments
	x = self._stack.pop()
	self._stack.push(~x)


    def bc_call(self, arg):
	"""Call global function.

	"arg" arguments to the function are expected on the stack
	(pushed in the same order that they are listed in the function
	parameter list), followed by a reference to the symbol that
	represents the name of the function (this is stored in the
	literals array and pushed using 03N).  Although the stack
	description indicates that the instruction itself pushes
	nothing onto the stack, remember that all NewtonScript
	functions conventionally push a single item onto the stack
	before returning.

	Example:

	max( foo, bar ) translates to:

	7B	push foo  (assuming it to be the first parameter of the
			   function) 
	7C	push bar  (assuming it to be the second)
	18	push 'max (assuming it to be the first literal in the
			   function)
	2A	Call a global function with two arguments.

	"""

	pass


    def bc_invoke(self, arg):
	"""Call function with 

	This directly translates the NewtonScript <#007F>call ... with
	() construct.  "arg" arguments to the function are expected
	on the stack (pushed in the same order that they are listed in
	the function parameter list), followed by a reference to a
	CodeBlock frame that has had the BC_SELF instruction called on
	it.  As before, remember that all NewtonScript functions
	conventionally push a single item onto the stack before
	returning.

	Example:

	call func(foo) return foo with (1) translates to:

	24	push 1
	18	push func (assuming it to be the first literal in the
			   containing function)
		Note that inline function definitions like this are
		encoded as a CodeBlock frame in the literals array of
		the containing function.
	04	Form a closure for the func (?)
	31	Call ... with passing one argument.

	"""

	pass


    def bc_send(self, arg):
	"""Send Message.


	This directly translates the NewtonScript <#007F>: construct.  N
	arguments to the message are expected on the stack (pushed in the same
	order that they are listed in the function parameter list), followed
	by a reference to the receiving frame and then the symbol of the
	message.  Again, remember that all NewtonScript functions
	conventionally push a single item onto the stack before returning.

	Example:

	:Open() translates to:

	03	push self
	18	push 'Open (assuming it to be the first literal in
			    the function)
	38	Send message with zero parameters.

	"""

	pass


    def bc_send_if_defined(self, arg):
	"""

	This directly translates the NewtonScript <#007F>:? construct.
	"arg" arguments to the message are expected on the stack
	(pushed in the same order that they are listed in the function
	parameter list), followed by a reference to the receiving
	frame and then the symbol of the message to be sent if the
	given function exists.  If the function doesn't exist, Nil is
	placed on the stack else it's left up to the function to put
	something there.

	Example:

	child:?DoItIfYouCan() translates to:

	7B	push child (assuming it to be the first parameter of
			    the function)
	18	push 'DoItIfYouCan (assuming it to be the first literal
				    in the function)
	40	Send message with zero parameters if possible.

	"""

	pass


    def bc_resend(self, arg):
	"""

	This directly translates the NewtonScript <#007F>inherited:
	construct.  "arg" arguments to the message are expected on the
	stack (pushed in the same order that they are listed in the
	function parameter list), followed by the symbol of the
	message.  Again, remember that all NewtonScript functions
	conventionally push a single item onto the stack before
	returning.

	Example:

	inherited:Open() translates to:

	18	push 'Open (assuming it to be the first literal in the
			    function)
	48	Send message with zero parameters to inheritance parent.

	"""

	pass


    def bc_resend_if_defined(self, arg):
	"""

	This directly translates the NewtonScript <#007F>inherited:?
	construct.  "arg" arguments to the message are expected on the
	stack (pushed in the same order that they are listed in the
	function parameter list), followed by the symbol of the
	message to be sent if the given function exists.  If the
	function doesn't exist, Nil is placed on the stack else it's
	left up to the function to put something there.

	Example:

	inherited:?DoItIfYouCan() translates to:

	18	push 'DoItIfYouCan (assuming it to be the first literal
				    in the function)
	50	Send message with zero parameters to inheritance parent
		(if possible).

	"""

	pass



    def bc_branch(self, arg):
	"""

	This instruction causes execution to pass to location "arg",
	where "arg" is the offset from the start of the block of
	bytecode (i.e. gotos are absolute, not relative).

	"""

	pass


    def bc_branch_if_true(self, arg):
	"""

	X is removed from the stack and examined.  If it is not Nil,
	execution continues with the next instruction in sequence.  If
	it is Nil, execution passes to location "arg" where "arg" is
	the offset from the start of the block of bytecode.

	"""

	pass


    def bc_branch_if_false(self, arg):
	"""

	X is removed from the stack and examined.  If it is Nil,
	execution continues with the next instruction in sequence.  If
	it is not Nil, execution passes to location "arg" where "arg"
	is the offset from the start of the block of bytecode.

	"""

	pass


    def bc_make_frame(self, arg):
	"""

	This bytecode creates a frame given "arg" slot values on the
	stack plus a frame map describing the slots in the frame.  The
	frame map is an array whose first item is Nil and then each
	subsequent item is a reference to a symbol giving the name of
	the corresponding slot in the frame being created.  The values
	are pushed onto the stack in the order of the slot names in
	the frame map.  If this is as clear as mud, maybe the example
	will help:

	Example:

	local fr := { size: 4, count: 7 } translates to:

	27 00 10	push 4
	27 00 1C	push 7
	18	push frame map (assuming it to be the first literal
				in the CodeBlock)
		The frame map is an array thus: [Nil, 'size, 'count]
	82	Create a frame with two slots
	A6	Store in fr (assumed to be the 6th slot in argFrame)

	A final NB.  The frame map array isn't a normal array of class
	Array.  It's class is a small integer, whose value seems to be
	made up from three possible flags.  I'm investigating the
	meaning of these flags at the moment.

	"""

	pass


    def bc_make_array(self, arg):
	"""

	This bytecode creates an array given N element values on the
	stack plus a reference to a symbol defining the class of the
	array.  The class of the array is specified in NewtonScript
	with an initial <#007F><class>: inside the array, e.g.:
	[stepChildren:].  If no class is specified, then the default
	class of Array is used.

	In the special case of N being 0xFFFF, the instruction takes a single
	NewtonScript integer from the stack instead of N element values and
	creates an empty array (i.e. each element is Nil) with the given
	number of elements.

	Example:

	local foo := [ 4, 7 ] translates to:

	27 00 10	push 4
	27 00 1C	push 7
	18	push 'Array (assuming it to be the first literal
			     in the CodeBlock)
	8A	Create an array with two elements
	A5	Store in foo (assumed to be the 5th slot in argFrame)

	"""

	pass


    def bc_add_array_slot(self, arg):
	"""

	This implements the AddArraySlot function e.g. AddArraySlot(
	foo, 2 ). Y is added as the last element of the array X.

	"""
	pass


    def bc_stringer(self, arg):  # index 22
	"""

	This takes an array of strings and combines them into a single
	string.  This function is used to implement the NewtonScript
	operators <#007F>& and <#007F>&&.  <#007F>&& is currently done
	by inserting an extra single space string into the array of
	strings to be translated.

	Example:

	fred && "me" translates to:

	70		push value of fred ('fred is the first literal)
	19 1A 1B	push " " (second), "me" (third) and 'Array (fourth)
	8B		Make an array from 3 parameters and a class symbol
	C7 00 16	Make a string from the array

	"""
	pass


    def bc_aref(self, arg):
	"""

	This pushes the array element Y of array X.  Y should be an integer.

	"""

	pass


    def bc_set_aref(self, arg):
	"""

	This assignes value Z to the pushes the array element Y of
	array X i.e.  X[Y] := Z.  Y should be an integer.

	"""

	pass


    def bc_length(self, arg):
	"""

	This implements the length function.

	"""
	pass


    def bc_clone(self, arg):   # index 19
	"""

	This implements the clone function.

	"""
	pass


    def bc_set_class(self, arg):    # index 20
	"""

	This implements the SetClass function e.g. SetClass( data,
	'Binary ).  X is a complex data structure (frame, array or
	data block); SetClass sets the class of that structure and
	then returns a reference to the now modified structure.

	"""
	pass


    def bc_class_of(self, arg):  # index 24
	"""Implements the NewtonScript ClassOf function.

	x = pop(); push (ClassOf(x))
	"""

	x = self._stack.pop()
	self._stack.push(x.ClassOf())


    def bc_has_path(self, arg):  # index 23
	"""

	This implements the exists operator, but only when checking
	for the existance of a slot in a frame (e.g. foo.bar exists).
	When checking for the existance of a variable, a function call
	(bytecode 05N) to the function HasVar is used instead.

	"""
	pass


    def bc_new_handlers(self, arg):
	"""

	This takes N pairs of an exception symbol and an offset into the
	bytecode to jump to if that exception occurs.  This is used to encode
	the high level try ... onexception construct.  Note that unlike all
	the goto bytecodes, the offset is encoded as a standard NewtonScript
	integer.  At the end of both the normal and the exception code, the
	bytecode 007 000 007 is executed - I guess it clears whatever state
	was set up.

	Example:

	try 1/0 onexception |evt.ex.div0| do nil becomes:

	18		push '|evt.ex.div0| (first literal)
	27 00 40	push integer 16
	C9		onexception '|evt.ex.div0| goto 16
	24 20		push 1, push 0
	C7 00 08	1/0
	07 00 07	End exception block (?)
	5F 00 14	Goto 20 (i.e. the end of this block)
	16: 22		Push Nil
	07 00 07	End exception block (?)

	"""
	pass


    def bc_new_iterator(self, arg):
	"""

	This is the fundamental instruction used to implement all
	NewtonScript foreach loops.  If the Boolean is Nil it begins a
	normal foreach loop and if it is True it begins a foreach
	deeply loop.  The iterator returned is then passed to the
	other two related instructions 005 (foreach next) and 006
	(foreach complete).  The iterator itself is an array
	containing various useful state information about what is
	being iterated over and how far the iteration has progressed.

	005 (foreach next) takes the iterator and modifies it so that
	the next item in the iteration becomes the current item.  006
	(foreach complete) checks the state of the iterator and
	returns True if all items have been iterated over and Nil if
	not.

	So, a foreach loop is implemented in bytecode as:

	??		push thing being iterated over
	27<#0102>00<#0102>1A/22	push True or Nil (for deeply or not)
	C7 00 11	Create foreach iterator
	A?		Assign it to a local variable.
	...		Perform other initialisation (depends on loop type)
	5F ?? ??	Goto end of loop check (Y:)
	X: ...		Code executed each time round the iteration.
	7?		Push the local variable referencing the iterator
	05		And move to the next thing to be iterated
	Y: 7?		Push the iterator again
	06		And check if we've finished iterating
	6F ?? ??	Go round the loop again if not (X:)


	"""
	pass


    def bc_incr_var(self, arg):
	"""

	This bytecode adds the given integer increment to the local
	variable in the Nth slot of the CodeBlock's argFrame and the
	returns both the increment and the new value of the local
	variable.  This instruction is used internally as part of the
	coding of for loops.

	"""

	pass


    def bc_branch_if_loop_not_done(self, arg):
	"""

	This instruction is placed at the end of a for loop construct.
	It takes the increment, current value of the counter (after
	having been incremented by 26N) and the limit value of the
	loop.  If the counter value now exceeds the loop limit,
	execution continues with the next instruction in sequence.  If
	it does not, execution passes to location N where N is the
	offset from the start of the block of bytecode.  This
	instruction is used internally as part of the coding of for
	loops.

	"""

	pass


#############################################################################
#
#   primitive function implementations
#


#############################################################################

def HasSlot(f, s):
    """Return true iff frame *f* has slot named *s*"""
    return f.has_key(s)


def GetSlot(f, s):
    """Return the value of the slot *s* from frame *f*"""
    return f[s]


def SetSlot(f, s, v):
    """Set slot *s* in frame *f* to value *v*"""
    f[s] = v
    return

def hword2text(hword):
    """Convert half word to printable string.

    *hword*  -- (integer) Python integer, sized < 16 bits
    Returns  -- (string) printable representation of the 16 bit quantity"""

    if is_int(hword):
	return str(hword >> 2)

    elif is_nil(hword):
	return "nil"

    elif is_true(hword):
	return "True"

    elif is_unicode(hword):
	return "$%s" % chr(hword2unicode(hword))

    elif is_magic(hword):
	return "@%d" % (hword >> 2)

    else:
	return "%04x" % hword


def hword2int(hword):
    """Convert half word to Python integer.

    "hword"         two Python characters, in ASCII (ie. 16 bits)
    Return value    Python integer, 14 bit range

    NewtonScript stores integers with the lowest two bits set to zero
    to indicate that it is an integer.  Thus, we get a 14 bit 2's
    complement number from a 16bit half word.

    This format is common in long format bytecodes."""

    if len(hword) != 2:
	raise TypeError, "'hword' must be two character string"

    hval = (ord(hword[0]) << 8) + ord(hword[1])
    i = (hval & 0xffff) >> 2
    if i & 0x2000:
	return -((~(i - 1)) & 0x3fff)
    else:
	return i
	

def is_int(word):
    """Is this word an integer?"""

    return (word & 3 == 0)


def is_pointer(word):
    """Is this word a pointer?"""

    return (word & 3 == 1)


def is_unique(word):
    """Is this word a unique system value?"""

    return (word & 3 == 2)


def is_magic(word):
    """Is this word a magic pointer?"""

    return (word & 3 == 3)



def is_nil(word):
    """Is this word "nil"?"""

    return (word == 2)


def is_true(word):
    """Is this word "True"?"""

    return (word == 0x1a)


def is_unicode(word):
    """Is this word a Unicode constant?"""

    return (word & 7 == 6)


#############################################################################

def main():
    """Test"""

    mem = "\x20" \
	  "\x22" \
	  "\x23" \
	  "\x24" \
	  "\x27\x00\x1a" \
	  "\x27\x02\x86" \
	  "\x27\xff\xf8" \
	  "\000" \
	  "\001" \
	  "\007\000\007"


    i = None #fixme:  NewtonVM(mem)
    i.go()


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
